using UnityEngine;
using System.Collections;
using System.Runtime.InteropServices;

namespace hiveAdiz
{
	public class AdizBanner
	{

		private static AdizBanner instance = null; 

		private AdizBanner() {}

		~AdizBanner() {
			//
		}

		private string adKey = "";

		public static AdizBanner Initialize(BannerSize size, AdEventHandlers handler) {

			string defaultKey = AdizCallbackManager.defaultBannerKey;
			if(string.IsNullOrEmpty(defaultKey)) { 
				AdizLogger.d("adKey is null or empty.");
				return null; 
			}

			instance = new AdizBanner();
			instance.internalInit(defaultKey, size, handler);
			return instance;
		}

		public static AdizBanner Initialize(string hiveAdKey, BannerSize size, AdEventHandlers handler) {

			if(string.IsNullOrEmpty(hiveAdKey)) { 
				AdizLogger.d("adKey is null or empty.");
				return null; 
			}

			instance = new AdizBanner();
			instance.internalInit(hiveAdKey, size, handler);
			return instance;
		}

		public static AdizBanner InitializeWithPlacementId(string placementId, BannerSize size, AdEventHandlers handler) {

			if(string.IsNullOrEmpty(placementId)) { 
				AdizLogger.d("placementId is null or empty.");
				return null; 
			}

			string hiveAdKey = AdizUnityNativeCaller.Instance.GetHiveAdKeyFromPlacementId(placementId);
			if(string.IsNullOrEmpty(hiveAdKey)) { 
				AdizLogger.d("placementId is invalid");
				return null; 
			}

			instance = new AdizBanner();
			instance.internalInit(hiveAdKey, size, handler);
			return instance;
		}

		private void release() {
			if(AdizCallbackManager.callbackHandler.ContainsKey(adKey)) {
				AdizCallbackManager.callbackHandler.Remove(adKey);	
			}

			instance = null;
		}
		
	#if UNITY_ANDROID && !UNITY_EDITOR
		
		private AndroidJavaObject currentActivity = null;
		private AndroidJavaObject hiveAdizPlugin = null;

		private void internalInit(string hiveAdKey, BannerSize size, AdEventHandlers handler) {
			AdizLogger.d("AdizBanner Init "+hiveAdKey);

			AndroidJavaClass jc = new AndroidJavaClass("com.unity3d.player.UnityPlayer");
			currentActivity = jc.GetStatic<AndroidJavaObject>("currentActivity");
			hiveAdizPlugin = new AndroidJavaObject("com.hive.adiz.plugin.AdizUnityBanner");

			adKey = hiveAdKey;
			hiveAdizPlugin.Call("initialize", currentActivity, hiveAdKey, (int) size);
			
			if(AdizCallbackManager.callbackHandler.ContainsKey(hiveAdKey)) {
				AdizCallbackManager.callbackHandler.Remove(hiveAdKey);	
			}
			AdizCallbackManager.callbackHandler.Add(hiveAdKey, handler);
		}

		public void Load() {
			AdizLogger.d("AdizBanner Load");
			hiveAdizPlugin.Call("load");
		}

		public void Show() {
			AdizLogger.d("AdizBanner Show");
			hiveAdizPlugin.Call("show");
		}

		public void Show(PositionType position) {
			int posNum = (int) position;
			hiveAdizPlugin.Call("show", posNum);
		}

		public void Destroy() {
			hiveAdizPlugin.Call("destroy");
			release();
		}

		public bool IsInitialized() {
			return hiveAdizPlugin.Call<bool>("isInitialized");
		}

		public bool IsLoaded() {
			return hiveAdizPlugin.Call<bool>("isLoaded");
		}

	#elif UNITY_IOS && !UNITY_EDITOR

		[DllImport("__Internal")]
		private static extern void Adiz_BannerInitialize(string hiveAdKey, int size);

		[DllImport("__Internal")]
		private static extern void Adiz_BannerLoad(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern void Adiz_BannerShow(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern void Adiz_BannerShowPosition(string hiveAdKey, int position);

		[DllImport("__Internal")]
		private static extern void Adiz_BannerDestroy(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern bool Adiz_BannerIsInitialized(string hiveAdKey);

		[DllImport("__Internal")]
		private static extern bool Adiz_BannerIsLoaded(string hiveAdKey);

		private void internalInit(string hiveAdKey, BannerSize size, AdEventHandlers handler) {
			AdizLogger.d("AdizBanner Init "+hiveAdKey);
			adKey = hiveAdKey;
			
			if(AdizCallbackManager.callbackHandler.ContainsKey(hiveAdKey)) {
				AdizCallbackManager.callbackHandler.Remove(hiveAdKey);	
			}
			AdizCallbackManager.callbackHandler.Add(hiveAdKey, handler);

			Adiz_BannerInitialize(hiveAdKey, (int) size);
		}

		public void Load() {
			AdizLogger.d("AdizBanner Load");
			Adiz_BannerLoad(adKey);
		}

		public void Show() {
			AdizLogger.d("AdizBanner Show");
			Adiz_BannerShow(adKey);
		}

		public void Show(PositionType position) {
			AdizLogger.d("AdizBanner Show Position");
			Adiz_BannerShowPosition(adKey, (int) position);
		}

		public void Destroy() {
			Adiz_BannerDestroy(adKey);
			release();
		}

		public bool IsInitialized() {
			return Adiz_BannerIsInitialized(adKey);
		}

		public bool IsLoaded() {
			return Adiz_BannerIsLoaded(adKey);
		}
	#else

		private void internalInit(string hiveAdKey, BannerSize size, AdEventHandlers handler) {
			
		}

		public void Load() {
			
		}

		public void Show() {
			
		}

		public void Show(PositionType position) {
			
		}

		public void Destroy() {
			
		}
		
		public bool IsInitialized() {
			return false;
		}
		
		public bool IsLoaded() {
			return false;
		}
	#endif
		
	}
}